# -*- coding: UTF-8 -*-
"""Scene related classes"""
import blindstation.cblind
from blindstation.utils import BSCError, BSError
from blindstation.surface import Surface, ImageSurface
from blindstation.event import Event, EventCode, UserCode, KeyCode, KeyMod
from blindstation.font import FontServer
import weakref

class Scene(object):
    """A Scene is a major sequence of a Game"""

    def __init__(self, game, name, state = None):
        self.next = self
        self.nextargs = []
        self.nextkeywords = {}
        self.timer_list = []
        self.widget_list = []
        self.game = game
        self.__focus__ = None
        self.dirty = True
        self.screen = self.game().screen
        self.lang = self.game().lang
        self.fontserver = FontServer()
        self.__color__ = [255, 0, 0, 255]
        self.__name__ = name
        self.__stylesheet__ = None
        self.__navigator__ = None
        self.__background__ = None
        self.__backsurface__ = None
        self.__state__ = None
        self.__zoom_mode__ = False
        if state != None:
            self.state = state

    def __del__(self):
        pass

    def register_widget(self, widget):
        for id in range(len(self.widget_list)):
            if self.widget_list[id] is None:
                self.widget_list[id] = weakref.ref(widget)
                break
        else:
            self.widget_list.append(weakref.ref(widget))
            id = len(self.widget_list) - 1
        if self.__focus__ is None and widget.accept_focus and not widget.hidden:
            self.__focus__ = id
        return id

    def unregister_widget(self, id):
        self.widget_list[id] = None
        self.dirty = True

    def register_timer(self, timer):
        for id in range(len(self.timer_list)):
            if self.timer_list[id] is None:
                self.timer_list[id] = timer
                break
        else:
            self.timer_list.append(timer)
            id = len(self.timer_list) - 1
        return id

    def unregister_timer(self, id):
        self.timer_list[id] = None

    def focus_prev(self):
        search_list = range(len(self.widget_list))
        if self.__focus__ is not None:
            search_list = search_list[self.__focus__ + 1 : ] + search_list[0 : self.__focus__]
        search_list.reverse()
        for index in search_list:
            if self.widget_list[index] is not None and self.widget_list[index]().accept_focus and not self.widget_list[index]().hidden:
                break
        else:
            index = None
        if index is not None:
            self.set_focus(index)

    def focus_next(self):
        search_list = range(len(self.widget_list))
        if self.__focus__ is not None:
            search_list = search_list[self.__focus__ + 1 :] + search_list[0 : self.__focus__]
        for index in search_list:
            if self.widget_list[index] is not None and self.widget_list[index]().accept_focus and not self.widget_list[index]().hidden:
                break
        else:
            index = None
        if index is not None:
            self.set_focus(index)

    def create(self):
        self.surface = Surface(self.screen.w, self.screen.h)
        self.surface.fill(self.__color__)
        if self.__background__ is not None and self.__backsurface__ is None:
            self.__backsurface__ = ImageSurface(self.__background__)

    def update(self):
        if self.__zoom_mode__:
            widget = self.widget_list[self.__focus__]
            if widget is None:
                return
            if widget() is None:
                return
            else:
                widget = widget()
            if self.dirty or widget.dirty:
                widget.create(2)
                widget.update(2)
                widget.dirty = False
                self.screen.update(0, 0, self.screen.w, self.screen.h)
            return
        if self.dirty:
            self.create()
            self.surface.blit(0, 0, self.surface.w, self.surface.h)
            if self.__backsurface__ is not None:
                self.__backsurface__.blit(0, 0, self.__backsurface__.w, self.__backsurface__.h)
            for widget in self.widget_list:
                if widget is not None:
                    if widget().dirty:
                        if self.widget_list[self.__focus__] == widget:
                            widget().create(1)
                            widget().update(1)
                        else:
                            widget().create(0)
                            widget().update(0)
                    else:
                        if self.widget_list[self.__focus__] == widget:
                            widget().update(1)
                        else:
                            widget().update(0)
            self.screen.update(0, 0, 0, 0)
            self.dirty = False
        else:
            for widget in self.widget_list:
                if widget is not None and widget().dirty and not widget().hidden:
                    self.surface.blitpart(widget().rect[0], widget().rect[1], widget().rect[0], widget().rect[1], widget().rect[2], widget().rect[3])
                    if self.__backsurface__ is not None:
                        if widget().rect[2] < self.__backsurface__.w:
                            w = widget().rect[2]
                        else:
                            w = self.__backsurface__.w
                        if widget().rect[3] < self.__backsurface__.h:
                            h = widget().rect[3]
                        else:
                            h = self.__backsurface__.h
                        self.__backsurface__.blitpart(widget().rect[0], widget().rect[1], widget().rect[0], widget().rect[1], w, h)
                    if self.widget_list[self.__focus__] == widget:
                        widget().create(1)
                        widget().update(1)
                    else:
                        widget().create(0)
                        widget().update(0)
                    self.screen.update(widget().rect[0], widget().rect[1], widget().rect[2], widget().rect[3])

    def run(self, event):
        """run is a method lauched each time an event occurs. It must
        handle the event or pass it to other objects that could handle
        it."""
        handled = False
        if event.type == EventCode.KEYDOWN or\
           event.type == EventCode.KEYUP or\
           event.type == EventCode.JOYAXISMOTION or\
           event.type == EventCode.JOYBUTTONDOWN:
            if self.__navigator__ is not None:
                handled = self.__navigator__.run(event)
                if handled is True:
                    return True
        if event.type == EventCode.KEYUP:
            return True
        elif event.type == EventCode.KEYDOWN:
            if event.sym == KeyCode.RETURN:
                if self.__focus__ is not None:
                    widget = self.widget_list[self.__focus__]()
                    callback = widget.validate_callback
                    if callback is not None:
                        callback(widget.name, EventCode.KEYDOWN, KeyCode.RETURN)
            elif event.sym == KeyCode.TAB:
                if event.mod & KeyMod.SHIFT:
                    self.focus_prev()
                else:
                    self.focus_next()
                handled = True
            else:
                for widget in self.widget_list:
                    if widget is not None:
                        if widget().run(event):
                            handled = True
                            break
                else:
                    print "unhandled key pressed with code", event.sym, "and unicode", event.unicode
                    handled = False
        elif event.type == EventCode.USEREVENT:
            if event.code == UserCode.CTIMER:
                for timer in self.timer_list:
                    if timer is not None:
                    	if timer.id == event.id:
                    	    callback = timer.timer_callback
                    	    if callback is not None:
                    	        callback(timer.name)
                    	    break
                else:
                    raise BSError, "undefined timer triggerred"
            elif event.code == UserCode.BRAILLE or event.code == UserCode.BOARD:
                for widget in self.widget_list:
                    if widget is not None:
                        if widget().run(event):
                            handled = True
                            break
            else:
                print "unknown user event with code", event.code
        else:
            print "event with type", event.type
        return handled

    def get_color(self):
        """get the Scene background color"""
        return self.__color__

    def set_color(self, color):
        """set the background color with the red, green, blue and apha component
        values can be between 0-255 with a=0 for transparent and a=255 for solid"""
        if not len(color) == 4:
            raise TypeError, "Invalid color format"
        else:
            if(self.__color__ != color):
                self.__color__ = color
                self.dirty = True

    def get_stylesheet(self):
        if self.__stylesheet__ is None:
            scene_list = self.game().stylesheet.getElementsByTagName("scene")
            for scene in scene_list:
                if scene.getAttribute('name') == self.__name__:
                    self.__stylesheet__ = scene
                    break
            else:
                raise NameError, "Cannot find scene resource `" + self.__name__ + "`"
        return self.__stylesheet__

    def set_navigator(self, navigator):
        self.__navigator__ = navigator

    def set_background(self, background):
        if self.__background__ != background:
            self.__background__ = background
            self.__backsurface__ = None
            self.dirty = True

    def get_background(self, background):
        return self.__background__

    def set_state(self, name):
        state_list = self.stylesheet.getElementsByTagName("state")
        for state in state_list:
            if state.getAttribute('name') == name:
                break
        else:
            raise NameError, "Cannot find state resource `" + name\
                  + "` for widget", self.name
        self.__state__ = name
        color_list = state.getElementsByTagName("color")
        for color in color_list:
            self.color = [int(color.getAttribute('r')), int(color.getAttribute('g')),\
                          int(color.getAttribute('b')), int(color.getAttribute('a'))]
            break
        background_list = state.getElementsByTagName("background")
        for background in background_list:
            self.background = background.getAttribute('file')
            if background.getAttribute('lang') == self.lang:
                break

    def get_state(self):
        return self.__state__

    def set_focus(self, index):
        if self.widget_list[index] is not None and self.widget_list[index]().accept_focus and not self.widget_list[index]().hidden:
            self.widget_list[self.__focus__]().dirty = True
            self.widget_list[index]().dirty = True
            self.__focus__ = index
            widget = self.widget_list[self.__focus__]()
            if widget.focus_callback is not None:
                widget.focus_callback(widget.name)
    
    def get_focus(self):
        if self.__focus__ is not None:
            return self.widget_list[self.__focus__]()
        else:
            return None

    def set_zoom_mode(self, state):
        if self.__zoom_mode__ != state:
            self.dirty = True
            if self.focus is not None:
                self.focus.dirty = True
            self.__zoom_mode__ = state
            if self.__zoom_mode__:
                self.__zoomfont__ = self.fontserver.get_font(self.game().__zoomfont__, self.game().__zoomfontsize__)
            else:
                del self.__zoomfont__

    def get_zoom_mode(self):
        return self.__zoom_mode__

    color = property(get_color, set_color, None,
                      "The backgroung color attribut")
    stylesheet = property(get_stylesheet, None, None,
                      "Part of the stylesheet related to this scene")
    navigator = property(None, set_navigator, None,
                      "The navigator component of the scene")
    background = property(get_background, set_background, None,
                    "Background picture to be displayed by the widget")
    state = property(get_state, set_state, None,
                    "State of the widget relating to the xml resource file")
    focus = property(get_focus, set_focus, None,
                    "Current widget with focus")
    zoom_mode = property(get_zoom_mode, set_zoom_mode, None,
                    "Change the zoom mode")
