# -*- coding: UTF-8 -*-
"""Game related classes"""
import blindstation.cblind
import xml.dom.minidom
import ConfigParser, os
import types, string
import weakref
from blindstation.utils import BSCError, BSError
from blindstation.surface import Surface, Screen
from blindstation.event import Event, EventCode

class Game(object):
    """The base class to initialise the BlindStation"""
    def __init__(self, scene, stylesheet, sceneargs = [], videomode = 0, lang = 'en'):
        """Initialise the game.
        The videomode is either:
        0: for 640x480
        1: for 800x600
        2: for 1024x768"""
        if type(sceneargs) != types.ListType:
            raise TypeError, "sceneargs must be a list"
        self.cgame = blindstation.cblind.CGame_init()
        if self.cgame is None:
            raise BSCError, 'CGame_init failed'
        if videomode == 0:
            self.__w__, self.__h__ = 640, 480
        elif videomode == 1:
            self.__w__, self.__h__ = 800, 600
        elif videomode == 2:
            self.__w__, self.__h__ = 1024, 768
        else:
            raise BSError, "Invalid video mode"
        config = ConfigParser.ConfigParser()
        config.readfp(open('blindstation/defaults.cfg'))
        config.read(['user.cfg',
                     os.path.expanduser('~/.blindstation.cfg'),
                     os.path.expanduser('~/blindstation.cfg')])
        self.__zoomfont__ = config.get("Zoom mode", "font")
        self.__zoomfontsize__ = config.getint("Zoom mode", "fontsize")
        self.__zoomcolor__ = map(int, string.split(config.get("Zoom mode", "color"), ',', 2)) + [255]
        self.__zoomtextcolor__ = map(int, string.split(config.get("Zoom mode", "textcolor"), ',', 2))
        self.__zoomshowpictures__ = config.getboolean("Zoom mode", "showpictures")
        self.__fullscreen__ = config.getboolean("General", "fullscreen")
        self.screen = Screen(self.__w__, self.__h__, fullscreen = self.__fullscreen__)
        self.lang = lang
        self.stylesheet = xml.dom.minidom.parse(stylesheet)
        sceneargs.insert(0, weakref.ref(self))
        self.scene = apply(scene, sceneargs)

    def __del__(self):
        "Game destructor"
        del self.scene
        self.stylesheet.unlink()
        Game.rootwindow = None
        if blindstation.cblind.CGame_del(self.cgame) == 0:
            raise BSCError, 'CGame_del failed'

    def run(self):
        """Process the events until game end"""
        while(1):
            handled = False
            self.scene.update()
            cevent = blindstation.cblind.CGame_run(self.cgame)
            event = Event(cevent)
            if event is None:
                raise BSCError, 'CGame_run failed'
            if event.type == EventCode.NOEVENT:
                continue
            elif event.type == EventCode.QUIT:
                break
            elif event.type == EventCode.KEYDOWN and event.sym == 178:
                break
            else:
                self.scene.run(event)
                if self.scene.next == self.scene:
                    continue
                elif self.scene.next == None:
                    break
                else:
                    next = self.scene.next
                    self.scene.next = None
                    if self.scene.nextargs != []:
                        nextargs = self.scene.nextargs
                    else:
                        nextargs = []
                    if self.scene.nextkeywords != []:
                        nextkeywords = self.scene.nextkeywords
                    else:
                        nextkeywords = {}
                    del self.scene
                    if blindstation.cblind.CGame_EmptyEvents(self.cgame) == 0:
                        raise BSCError, 'CGame_EmptyEvents failed'
                    nextargs.insert(0, weakref.ref(self))
                    self.scene = apply(next, nextargs, nextkeywords)

    def exit(self):
        "Quit the running Game"
        if blindstation.cblind.CGame_exit(self.cgame) == 0:
            raise BSCError, 'CGame_del failed'

    def set_fullscreen(self, full):
        "Set the platform in full screen mode"
        if self.__fullscreen__ != full:
            self.screen = Screen(self.__w__, self.__h__, 0, full)
            self.scene.__dirty__ = True
            self.__fullscreen__ = full

    def setAutorepeat(self, delay, interval):
        """Change the autorepeat time of keys.
        delay: the delay before repeating
        interval: the interval between 2 repetitions"""
        if blindstation.cblind.CGame_setAutorepeat(self.cgame, delay, interval) == 0:
            raise BSCError, 'CGame_setAutorepeat failed'

    def reInitBraille(self):
        """reinitialise the braille subsystem"""
        if blindstation.cblind.CGame_reInitBraille(self.cgame) == 0:
            raise BSCError, 'CGame_reInitBraille failed'

    def reInitBoard(self):
        """reinitialise the board subsystem"""
        if blindstation.cblind.CGame_reInitBoard(self.cgame) == 0:
            raise BSCError, 'CGame_reInitBoard failed'
