# -*- coding: UTF-8 -*-
"""Sound related classes"""
import blindstation.cblind
import blindstation.game
import weakref
from blindstation.utils import BSCError, BSError, WeakMethod, BSSourceLimit
from blindstation.timer import Timer

class Listener(object):
    """A Listener represents the player and his position"""

    def __init__(self):
        self.__gain__ = 1.0
        if blindstation.cblind.CListener_init() == 0:
            raise BSCError, 'CListener_init failed'

    def setOrientation(self, forward_x, forward_y, forward_z, up_x, up_y, up_z):
        "Set the orientation of the Listener according to the given vector"
        if blindstation.cblind.CListener_setOrientation(forward_x, forward_y, forward_z, up_x, up_y, up_z) == 0:
            raise BSCError, 'CListener_setOrientation failed'

    def setPosition(self, x, y, z):
        "Set the position of the Listener to coordinates x, y, z"
        if blindstation.cblind.CListener_setPosition(x, y, z) == 0:
            raise BSCError, 'CListener_setPosition failed'

    def setVelocity(self, x, y, z):
        "Set the velocity of the Listener according to the vector (x, y, z)"
        if blindstation.cblind.CListener_setVelocity(x, y, z) == 0:
            raise BSCError, 'CListener_setVelocity failed'

    def set_gain(self, vol):
        """Set the global gain which affect the volume of all sounds -"
        "A value of 0 means no sound. 1 means default vol. >1 means amplification"""
        if blindstation.cblind.CListener_setGain(vol) == 0:
            raise BSCError, 'CListener_setGain failed'
        else:
            self.__gain__ = vol

    def get_gain(self, vol):
        return self.__gain__
    
    gain = property(get_gain, set_gain, None,
                    """gain which affect the volume of all sounds -"
                    "A value of 0 means no sound."
                    "1 means default volume."
                    ">1 means amplification""")


class Source(Timer):
    """A Source represents a position from which a sound can be played"""

    def __init__(self, scene, name):
        "A source must be initialised with its name"
        Timer.__init__(self, scene, name)
        self.csource = blindstation.cblind.CSource_init()
        if self.csource is None:
            # raise BSCError, 'CSource_init failed'
            raise BSSourceLimit, 'Source limit achieved'
        self.sound_queue = []
        self.sound_current = 0
        self.looping = 0
        self.timer_callback = self.__internal_callback__
        self.__source_callback__ = None

    def __del__(self):
        "Source destructor"
        Timer.__del__(self)
        if blindstation.cblind.CSource_del(self.csource) == 0:
            raise BSCError, 'CSource_del failed'

    def __internal_callback__(self, name):
        if len(self.sound_queue) == 0:
            raise BSError, 'empty sound queue'
        if len(self.sound_queue) > self.sound_current + 1:
            self.sound_current += 1
            self.play(self.sound_current)
        else:
            self.setTime(0)
            callback = self.source_callback
            if callback is not None:
                callback(self.name)
            if self.looping == 1:
                self.play(0)

    def queue(self, name):
        """Add the sound resource with the given name to the queue
        of sounds to be played"""
        lang = self.scene().lang
        sound_list = self.scene().stylesheet.getElementsByTagName("sound")
        for sound in sound_list:
            if sound.getAttribute('name') == name:
                break
        else:
            raise NameError, "Cannot find sound resource `" + name + "`"
        file_list = sound.getElementsByTagName("file")
        for file in file_list:
            filename = file.getAttribute('name')
            if file.getAttribute('lang') == lang:
                break
        self.sound_queue.append(filename)

    def empty(self):
        """Remove all the sounds from the sound queue"""
        self.stop()
        self.sound_queue = []
        self.sound_current = 0

    def play(self, sound = 0):
        "Start playing sounds in the queue"
        self.sound_current = sound
        if len(self.sound_queue) <= 0 :
            raise BSError, "Empty sound queue"
        time = blindstation.cblind.CSource_Play(self.csource, self.sound_queue[self.sound_current].encode('ascii'))
        if time == -1:
            raise BSCError, 'CSource_Play failed'
        if time > 0:
            self.setTime(time)
        
    def stop(self):
        "Stop sound from this source"
        self.setTime(0)
        if blindstation.cblind.CSource_Stop(self.csource) == 0:
            raise BSCError, 'CSource_Stop failed'

    def setVolume(self, vol):
        """Set the volume of this source
        with volume a float between 0.0 and 1.0
        """
        if blindstation.cblind.CSource_setVolume(self.csource, vol) == 0:
            raise BSCError, 'CSource_setVolume failed'

    def setPosition(self, x, y, z):
        """Move the source to a new position
        at coordinates x, y, z
        """
        if blindstation.cblind.CSource_setPosition(self.csource, x, y, z) == 0:
            raise BSCError, 'CSource_setPosition failed'

    def setVelocity(self, x, y, z):
        """Change acceleration of this Source to a vector x, y, z"""
        if blindstation.cblind.CSource_setVelocity(self.csource, x, y, z) == 0:
            raise BSCError, 'CSource_setVelocity failed'

    def setSurround(self, state):
        "Source is surround if state = 1, not surround otherwise"
        if blindstation.cblind.CSource_setSurround(self.csource, state) == 0:
            raise BSCError, 'CSource_setSurround failed'
            
    def setLooping(self, loop):
        "Source should loop its sounds"
        self.looping = loop

    def isPlaying(self):
        "Return true if the source is playing, false otherwise"
        state = blindstation.cblind.CSource_isPlaying(self.csource)
        if state == -1:
            raise BSCError, 'CSource_isPlaying failed'
        else:
            return state

    def set_source_callback(self, callback):
        if callback is None:
            self.__source_callback__ = None
        else:
            self.__source_callback__ = WeakMethod(callback)

    def get_source_callback(self):
        if self.__source_callback__ is None:
            return None
        callback = self.__source_callback__()
        if callback is None:
            raise BSError, "the source callback has been garbage collected"
        return callback

    source_callback = property(get_source_callback, set_source_callback, None,
                              "Callback to be called when all the sounds have been played")
