# -*- coding: UTF-8 -*-
"""Navigator related classes"""
import blindstation.cblind
import weakref
from blindstation.utils import BSCError, BSError, WeakMethod
from blindstation.event import Event, EventCode, UserCode, KeyCode
from blindstation.timer import Timer

class Direction(object):
    def __init__(self, x = 0, y = 0):
        self.x = x
        self.y = y

class Navigator(object):
    """A Navigator is a component to navigate in a virtual world"""

    def __init__(self, scene, state = None):
        self.__olddir__ = Direction(0, 0)
        self.__dir__ = Direction(0, 0)
        self.__navigator_callback__ = None
        self.__timer__ = Timer(scene, "navigator_timer")
        self.__timer__.timer_callback = self.timer_callback
        self.__repeat__ = 300
        self.key = KeyCode.RETURN
        scene.navigator = self

    def run(self, event):
        if event.type == EventCode.JOYBUTTONDOWN:
            event.type = EventCode.KEYDOWN
            event.sym = self.key
            event.unicode = 0
            return False

        callback = self.__navigator_callback__()
        if callback is None:
            return False
        handled = False

        self.__olddir__.x = self.__dir__.x
        self.__olddir__.y = self.__dir__.y

        if event.type == EventCode.KEYDOWN:
            if event.sym == KeyCode.LEFT:
                self.__dir__.x = -1
                handled = True
            elif event.sym == KeyCode.RIGHT:
                self.__dir__.x = 1
                handled = True
            elif event.sym == KeyCode.UP:
                self.__dir__.y = 1
                handled = True
            elif event.sym == KeyCode.DOWN:
                self.__dir__.y = -1
                handled = True
            if handled:
                callback(self.__dir__)
                self.__timer__.setTime(self.__repeat__)
            return handled
        elif event.type == EventCode.KEYUP:
            if event.sym == KeyCode.LEFT or event.sym == KeyCode.RIGHT:
                self.__dir__.x = 0
                handled = True
            elif event.sym == KeyCode.UP or event.sym == KeyCode.DOWN:
                self.__dir__.y = 0
                handled = True
            if handled and self.__dir__.x == 0 and self.__dir__.y == 0:
                self.__timer__.setTime(0)
            return handled
        elif event.type == EventCode.JOYAXISMOTION:
            if event.which == 0:
                if event.axis == 0:
                    if event.value > 16000:
                        self.__dir__.x = 1
                    elif event.value < -16000:
                        self.__dir__.x = -1
                    else:
                        self.__dir__.x = 0
                elif event.axis == 1:
                    if event.value > 16000:
                        self.__dir__.y = -1
                    elif event.value < -16000:
                        self.__dir__.y = 1
                    else:
                        self.__dir__.y = 0
                if self.__dir__.x != self.__olddir__.x or self.__dir__.y != self.__olddir__.y:
                    if self.__dir__.x == 0 and self.__dir__.y == 0:
                        self.__timer__.setTime(0)
                    else:
                        self.__timer__.setTime(self.__repeat__)
                        callback(self.__dir__)
            return True
        else:
            return False

    def timer_callback(self, name):
        callback = self.__navigator_callback__()
        if callback is None:
            print "timer should not be set"
            return
        callback(self.__dir__)

    def set_navigator_callback(self, callback):
        if callback is None:
            self.__navigator_callback__ = None
        else:
            self.__navigator_callback__ = WeakMethod(callback)

    def get_navigator_callback(self):
        if self.__navigator_callback__ is None:
            return None
        callback = self.__navigator_callback__()
        if callback is None:
            raise BSError, "the navigator callback has been garbage collected"
        return callback

    def set_repeat(self, time):
        self.__repeat__ = time

    def get_repeat(self):
        return self.__repeat__

    navigator_callback = property(get_navigator_callback, set_navigator_callback, None,
                                  "Callback to be called by the navigator")
    repeat = property(get_repeat, set_repeat, None,
                      "Time to wait bfore repeating navigator events")
