# -*- coding: UTF-8 -*-
"""Surface related classes"""
import blindstation.cblind
from blindstation.utils import BSCError, BSError
from blindstation.font import Font

class Surface(object):
    """A surface represents some graphics that can be displayed"""

    def __init__(self, w, h):
        self.csurface = blindstation.cblind.CSurface_init(w, h)
        if self.csurface is None:
            raise BSCError, 'CSurface_init failed'

    def __del__(self):
        if blindstation.cblind.CSurface_del(self.csurface) == 0:
            raise BSCError, 'CSurface_del failed'

    def blit(self, x, y, w, h):
        if blindstation.cblind.CSurface_blit(self.csurface, x, y, w, h) == 0:
            raise BSCerror, 'CSurface_blit failed'

    def blitpart(self, srcx, srcy, destx, desty, w, h):
        if blindstation.cblind.CSurface_blitpart(self.csurface, srcx, srcy, destx, desty, w, h) == 0:
            raise BSCerror, 'CSurface_blitpart failed'

    def fill(self, color):
        if blindstation.cblind.CSurface_fill(self.csurface, color[0], color[1], color[2], color[3]) == 0:
            raise BSCerror, 'CSurface_fill failed'

    def get_w(self):
        return self.csurface.w

    def get_h(self):
        return self.csurface.h

    # Define a property.  The 'delete this attribute'
    # method is defined as None, so the attribute
    # can't be deleted.
    w = property(get_w, None, None,
                    "Width of this surface")
    h = property(get_h, None, None,
                    "Height of this surface")

class TextSurface(Surface):
    """represents a Surface with some text"""

    def __init__(self, w, h):
        Surface.__init__(self, w, h)
        self.linelength = 0

    def __del__(self):
        Surface.__del__(self)

    def filltext(self, text, color, font, position):
        #    A word-wrap function that preserves existing line breaks
        #    and most spaces in the text. Expects that existing line
        #    breaks are posix newlines (\n).
        lines = reduce(lambda line, word: '%s%s%s' %
                       (line,
                        ' \n'[(font.size_text(line[line.rfind('\n')+1:] +
                                              word.split('\n',1)[0]) >= self.w - 2 * 7)],
                        word),
                       text.split(' ')
                       ).split('\n')
        lines = lines[: int(self.h /font.height())]
        self.linelength = len(lines[0])
        for i in range(len(lines)):
            if lines[i] != '':
                if position == "center":
                    pos = (self.csurface.w - font.size_text(lines[i]))/2
                elif position == "right":
                    pos = self.csurface.w - font.size_text(lines[i]) - 4
                else:
                    pos = 4
                if blindstation.cblind.CSurface_filltext(self.csurface,
                                                         font.cfont,
                                                         color[0], color[1], color[2],
                                                         lines[i], pos,
                                                         4 + font.height() * i
                                                         ) == 0:
                    raise BSCError, 'CSurface_filltext failed'

class ImageSurface(Surface):
    def __init__(self, imagefile):
        self.csurface = blindstation.cblind.CSurface_newFromFile(imagefile.encode('ascii'))
        if self.csurface is None:
            raise BSCError, 'CSurface_newFromFile failed'

    def __del__(self):
        if blindstation.cblind.CSurface_del(self.csurface) == 0:
            raise BSCError, 'CSurface_del failed'

class Screen(Surface):
    def __init__(self, w, h, bpp = 0, fullscreen = 0):
        self.csurface = blindstation.cblind.Screen_setVideoMode(w, h, bpp, fullscreen)
        if self.csurface is None:
            raise BSCError, 'CSurface_setVideoMode failed'

    def __del__(self):
        pass

    def update(self, x, y, w, h):
        if blindstation.cblind.Screen_update(x, y, w, h) == 0:
            raise BSCerror, 'Screen_update failed'
